import { sql } from "drizzle-orm";
import { pgTable, text, varchar, serial, timestamp, jsonb, numeric, integer, boolean } from "drizzle-orm/pg-core";
import { createInsertSchema } from "drizzle-zod";
import { z } from "zod";
import { users } from "./models/auth";

// Export auth models (users, sessions tables for Replit Auth)
export * from "./models/auth";

export const assessments = pgTable("assessments", {
  id: serial("id").primaryKey(),
  email: text("email"),
  phone: text("phone"),
  timestamp: text("timestamp").notNull(),
  ipAddress: text("ip_address").notNull(),
  phase: integer("phase").notNull(),
  answers: jsonb("answers").notNull(),
  overallScore: numeric("overall_score").notNull(),
  dimensionScore: numeric("dimension_score").notNull(),
  riskScore: numeric("risk_score").notNull(),
  summary: text("summary").notNull(),
  createdAt: timestamp("created_at").defaultNow().notNull(),
});

export const insertAssessmentSchema = createInsertSchema(assessments).omit({
  id: true,
  createdAt: true,
});

export type InsertAssessment = z.infer<typeof insertAssessmentSchema>;
export type Assessment = typeof assessments.$inferSelect;

// Leaderboard entries table
export const leaderboardEntries = pgTable("leaderboard_entries", {
  id: serial("id").primaryKey(),
  displayType: text("display_type").notNull(), // 'real', 'fancy', 'anonymous'
  displayName: text("display_name").notNull(),
  score: numeric("score").notNull(),
  phase: integer("phase").notNull(),
  isAspirational: boolean("is_aspirational").notNull(),
  timestamp: text("timestamp").notNull(),
  createdAt: timestamp("created_at").defaultNow().notNull(),
});

export const insertLeaderboardEntrySchema = createInsertSchema(leaderboardEntries).omit({
  id: true,
  createdAt: true,
});

export type InsertLeaderboardEntry = z.infer<typeof insertLeaderboardEntrySchema>;
export type LeaderboardEntry = typeof leaderboardEntries.$inferSelect;

// AI Coaching conversations and messages
export const conversations = pgTable("conversations", {
  id: serial("id").primaryKey(),
  title: text("title").notNull(),
  assessmentContext: jsonb("assessment_context"), // Stores assessment results for personalized coaching
  createdAt: timestamp("created_at").default(sql`CURRENT_TIMESTAMP`).notNull(),
});

export const messages = pgTable("messages", {
  id: serial("id").primaryKey(),
  conversationId: integer("conversation_id").notNull().references(() => conversations.id, { onDelete: "cascade" }),
  role: text("role").notNull(), // 'user' | 'assistant' | 'system'
  content: text("content").notNull(),
  createdAt: timestamp("created_at").default(sql`CURRENT_TIMESTAMP`).notNull(),
});

export const insertConversationSchema = createInsertSchema(conversations).omit({
  id: true,
  createdAt: true,
});

export const insertMessageSchema = createInsertSchema(messages).omit({
  id: true,
  createdAt: true,
});

export type Conversation = typeof conversations.$inferSelect;
export type InsertConversation = z.infer<typeof insertConversationSchema>;
export type Message = typeof messages.$inferSelect;
export type InsertMessage = z.infer<typeof insertMessageSchema>;

// Value Journey Assessments - stores full assessment results
export const valueJourneyAssessments = pgTable("value_journey_assessments", {
  id: serial("id").primaryKey(),
  sessionId: text("session_id").notNull(),
  stakeholder: text("stakeholder").notNull(),
  currentLevel: text("current_level").notNull(),
  aspirationalLevel: text("aspirational_level").notNull(),
  currentAnswers: jsonb("current_answers").notNull(),
  aspirationalAnswers: jsonb("aspirational_answers"),
  currentScore: integer("current_score").notNull(),
  aspirationalScore: integer("aspirational_score"),
  gapScore: integer("gap_score"),
  gleamsEarned: integer("gleams_earned").notNull(),
  alicornsEarned: numeric("alicorns_earned").notNull(),
  isComplete: boolean("is_complete").default(false).notNull(),
  createdAt: timestamp("created_at").defaultNow().notNull(),
});

export const insertValueJourneyAssessmentSchema = createInsertSchema(valueJourneyAssessments).omit({
  id: true,
  createdAt: true,
});

export type InsertValueJourneyAssessment = z.infer<typeof insertValueJourneyAssessmentSchema>;
export type ValueJourneyAssessment = typeof valueJourneyAssessments.$inferSelect;

// Contest Cases - for challenging assessment scores
export const contestCases = pgTable("contest_cases", {
  id: serial("id").primaryKey(),
  assessmentId: integer("assessment_id").notNull().references(() => valueJourneyAssessments.id),
  questionCode: text("question_code").notNull(),
  originalScore: integer("original_score").notNull(),
  contestedScore: integer("contested_score").notNull(),
  rationale: text("rationale").notNull(),
  evidenceLinks: text("evidence_links").array(),
  status: text("status").notNull().default("pending"), // pending, approved, rejected, partial
  adjudicatorNotes: text("adjudicator_notes"),
  finalScore: integer("final_score"),
  gleamsCost: integer("gleams_cost").notNull(),
  gleamsRefunded: integer("gleams_refunded").default(0),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  resolvedAt: timestamp("resolved_at"),
});

export const insertContestCaseSchema = createInsertSchema(contestCases).omit({
  id: true,
  createdAt: true,
  resolvedAt: true,
});

export type InsertContestCase = z.infer<typeof insertContestCaseSchema>;
export type ContestCase = typeof contestCases.$inferSelect;

// User Assessment Progress - tracks permanent stakeholder and overall progress
export const userAssessmentProgress = pgTable("user_assessment_progress", {
  id: serial("id").primaryKey(),
  userId: varchar("user_id").notNull().references(() => users.id),
  stakeholder: text("stakeholder").notNull(), // Permanent stakeholder choice
  additionalStakeholderRequests: text("additional_stakeholder_requests").array(), // Future requests for other stakeholders
  totalGleams: integer("total_gleams").default(0).notNull(),
  totalAlicorns: numeric("total_alicorns").default("0").notNull(),
  currentPauseState: jsonb("current_pause_state"), // { isPaused, pauseStartTime, breakDuration, alarmTime }
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

export const insertUserAssessmentProgressSchema = createInsertSchema(userAssessmentProgress).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export type InsertUserAssessmentProgress = z.infer<typeof insertUserAssessmentProgressSchema>;
export type UserAssessmentProgress = typeof userAssessmentProgress.$inferSelect;

// Level Completions - tracks individual level completions with timestamp and IP
export const levelCompletions = pgTable("level_completions", {
  id: serial("id").primaryKey(),
  userId: varchar("user_id").notNull().references(() => users.id),
  progressId: integer("progress_id").notNull().references(() => userAssessmentProgress.id),
  level: text("level").notNull(), // L0, L1, L2, etc.
  assessmentType: text("assessment_type").notNull(), // 'current' or 'aspirational'
  score: integer("score").notNull(),
  gleamsEarned: integer("gleams_earned").notNull(),
  alicornsEarned: numeric("alicorns_earned").notNull(),
  answers: jsonb("answers").notNull(),
  completedAt: timestamp("completed_at").defaultNow().notNull(),
  ipAddress: text("ip_address").notNull(),
});

export const insertLevelCompletionSchema = createInsertSchema(levelCompletions).omit({
  id: true,
  completedAt: true,
});

export type InsertLevelCompletion = z.infer<typeof insertLevelCompletionSchema>;
export type LevelCompletion = typeof levelCompletions.$inferSelect;

// Pause state type for TypeScript
export interface PauseState {
  isPaused: boolean;
  pauseStartTime: number | null;
  breakDuration: number; // in milliseconds
  alarmTime: number | null; // timestamp when alarm should trigger
  level: string; // which level was being assessed
  questionIndex: number; // where they left off
}

// User Suggestions - collected via AI chatbot
export const userSuggestions = pgTable("user_suggestions", {
  id: serial("id").primaryKey(),
  userId: varchar("user_id").references(() => users.id),
  sessionId: text("session_id"),
  suggestion: text("suggestion").notNull(),
  category: text("category").notNull().default("general"), // feature, bug, ux, content, other
  status: text("status").notNull().default("pending"), // pending, reviewed, implemented, declined
  gleamsAwarded: integer("gleams_awarded").default(0),
  ipAddress: text("ip_address"),
  createdAt: timestamp("created_at").defaultNow().notNull(),
});

export const insertUserSuggestionSchema = createInsertSchema(userSuggestions).omit({
  id: true,
  createdAt: true,
});

export type InsertUserSuggestion = z.infer<typeof insertUserSuggestionSchema>;
export type UserSuggestion = typeof userSuggestions.$inferSelect;

// Free Users - pre-registration user data storage
export const freeUsers = pgTable("free_users", {
  id: serial("id").primaryKey(),
  sessionId: text("session_id").notNull().unique(),
  email: text("email"),
  displayName: text("display_name"), // Fancy name
  primaryStakeholder: text("primary_stakeholder"), // Locked stakeholder
  additionalStakeholders: text("additional_stakeholders").array(), // Other roles added later
  currentLevel: text("current_level"), // Self-assessment level
  aspirationalLevel: text("aspirational_level"),
  totalGleams: integer("total_gleams").default(0).notNull(),
  totalAlicorns: numeric("total_alicorns").default("0").notNull(),
  completedLevels: text("completed_levels").array(), // ["L0", "L1", ...]
  levelScores: jsonb("level_scores"), // { "L0": { current: 85, aspirational: 90 }, ... }
  currentAnswers: jsonb("current_answers"), // Answers for current assessment
  aspirationalAnswers: jsonb("aspirational_answers"),
  isStakeholderLocked: boolean("is_stakeholder_locked").default(false),
  country: text("country"),
  sector: text("sector"),
  registrationStatus: text("registration_status").default("free").notNull(), // free, pending, registered
  i2uUserId: text("i2u_user_id"), // Linked i2u.ai user ID after registration
  ipAddress: text("ip_address"),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

export const insertFreeUserSchema = createInsertSchema(freeUsers).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export type InsertFreeUser = z.infer<typeof insertFreeUserSchema>;
export type FreeUser = typeof freeUsers.$inferSelect;

// User Ideas - each user can have multiple startup ideas
export const userIdeas = pgTable("user_ideas", {
  id: serial("id").primaryKey(),
  sessionId: text("session_id").notNull(), // Links to freeUsers.sessionId
  userId: varchar("user_id").references(() => users.id), // Links to registered users
  name: text("name").notNull(), // "My SaaS Platform", "HealthTech App", etc.
  description: text("description"), // Brief idea description
  sector: text("sector"), // Industry/sector for this idea
  stage: text("stage"), // "spark", "hunt", "build", "launch", etc.
  isActive: boolean("is_active").default(true).notNull(),
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

export const insertUserIdeaSchema = createInsertSchema(userIdeas).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export type InsertUserIdea = z.infer<typeof insertUserIdeaSchema>;
export type UserIdea = typeof userIdeas.$inferSelect;

// Idea Stakeholder Roles - links stakeholder roles to specific ideas
export const ideaRoles = pgTable("idea_roles", {
  id: serial("id").primaryKey(),
  ideaId: integer("idea_id").notNull().references(() => userIdeas.id, { onDelete: "cascade" }),
  sessionId: text("session_id").notNull(),
  stakeholder: text("stakeholder").notNull(), // founder, mentor, investor, talent, enabler, etc.
  isPrimary: boolean("is_primary").default(false).notNull(), // Primary role for this idea
  isActive: boolean("is_active").default(true).notNull(),
  createdAt: timestamp("created_at").defaultNow().notNull(),
});

export const insertIdeaRoleSchema = createInsertSchema(ideaRoles).omit({
  id: true,
  createdAt: true,
});

export type InsertIdeaRole = z.infer<typeof insertIdeaRoleSchema>;
export type IdeaRole = typeof ideaRoles.$inferSelect;

// Assessment Sessions - tracks progress for each idea+role+level combination
export const assessmentSessions = pgTable("assessment_sessions", {
  id: serial("id").primaryKey(),
  sessionId: text("session_id").notNull(),
  ideaId: integer("idea_id").references(() => userIdeas.id, { onDelete: "cascade" }),
  ideaRoleId: integer("idea_role_id").references(() => ideaRoles.id, { onDelete: "cascade" }),
  stakeholder: text("stakeholder").notNull(),
  level: text("level").notNull(), // L0, L1, L2, etc.
  assessmentType: text("assessment_type").notNull(), // 'current' or 'aspirational'
  status: text("status").notNull().default("in_progress"), // in_progress, paused, completed, abandoned
  currentQuestionIndex: integer("current_question_index").default(0).notNull(),
  answers: jsonb("answers"), // Partial answers saved
  score: integer("score"),
  gleamsEarned: integer("gleams_earned").default(0),
  alicornsEarned: numeric("alicorns_earned").default("0"),
  pausedAt: timestamp("paused_at"),
  pauseDuration: integer("pause_duration"), // Planned break duration in milliseconds
  scheduledReturnAt: timestamp("scheduled_return_at"), // Optional: when user plans to return
  exitReason: text("exit_reason"), // "break", "switch_idea", "switch_role", "done_for_now"
  startedAt: timestamp("started_at").defaultNow().notNull(),
  completedAt: timestamp("completed_at"),
  lastActivityAt: timestamp("last_activity_at").defaultNow().notNull(),
});

export const insertAssessmentSessionSchema = createInsertSchema(assessmentSessions).omit({
  id: true,
  startedAt: true,
  lastActivityAt: true,
});

export type InsertAssessmentSession = z.infer<typeof insertAssessmentSessionSchema>;
export type AssessmentSession = typeof assessmentSessions.$inferSelect;

// Enhanced Pause State type for TypeScript
export interface EnhancedPauseState {
  isPaused: boolean;
  pauseStartTime: number | null;
  breakDuration: number | null; // in milliseconds, null if no specific duration
  scheduledReturnTime: number | null; // timestamp when user plans to return
  alarmTime: number | null; // timestamp when alarm should trigger
  level: string;
  questionIndex: number;
  ideaId: number | null;
  ideaRoleId: number | null;
  exitReason: "break" | "switch_idea" | "switch_role" | "done_for_now" | null;
}

// User Sessions - tracks anonymous sessions with IP and microsecond timestamps for magic link auth
export const userSessions = pgTable("user_sessions", {
  id: serial("id").primaryKey(),
  sessionId: text("session_id").notNull().unique(),
  ipAddress: text("ip_address").notNull(),
  createdAtMicro: text("created_at_micro").notNull(), // ISO string with microsecond precision
  userAgent: text("user_agent"),
  displayName: text("display_name"), // Fancy name or real name for magic link validation
  lastAccessedAt: timestamp("last_accessed_at").defaultNow().notNull(),
  isActive: boolean("is_active").default(true).notNull(),
});

export const insertUserSessionSchema = createInsertSchema(userSessions).omit({
  id: true,
  lastAccessedAt: true,
});

export type InsertUserSession = z.infer<typeof insertUserSessionSchema>;
export type UserSession = typeof userSessions.$inferSelect;

// ========== ECOSYSTEM ENTITIES ==========

// Entity Types for the startup ecosystem
export const ENTITY_TYPES = [
  "startup",
  "investor_vc",
  "investor_angel", 
  "investor_pe",
  "investor_family_office",
  "investor_hedge_fund",
  "investor_sovereign",
  "mentor",
  "corporate",
  "accelerator",
  "incubator",
  "enabler",
  "facilitator",
  "talent",
  "influencer",
  "government",
  "ngo"
] as const;

export type EntityType = typeof ENTITY_TYPES[number];

// Sectors/Industries
export const SECTORS = [
  "AI/ML",
  "FinTech",
  "HealthTech",
  "EdTech",
  "CleanTech",
  "AgriTech",
  "PropTech",
  "InsurTech",
  "LegalTech",
  "HRTech",
  "MarTech",
  "DeepTech",
  "BioTech",
  "SpaceTech",
  "Cybersecurity",
  "E-commerce",
  "SaaS",
  "Gaming",
  "Web3/Crypto",
  "IoT",
  "Robotics",
  "Consumer",
  "Enterprise",
  "Marketplace",
  "Media/Content",
  "Travel/Hospitality",
  "Logistics",
  "Manufacturing",
  "Other"
] as const;

export type Sector = typeof SECTORS[number];

// Ecosystem Profiles - main profile for all entity types
export const ecosystemProfiles = pgTable("ecosystem_profiles", {
  id: serial("id").primaryKey(),
  sessionId: text("session_id"), // Link to freeUsers for user-generated profiles
  userId: varchar("user_id").references(() => users.id),
  entityType: text("entity_type").notNull(), // One of ENTITY_TYPES
  entitySubtype: text("entity_subtype"), // Additional classification (e.g., "Series A" for VC)
  name: text("name").notNull(), // Company/Person name
  displayName: text("display_name"), // Public display name
  tagline: text("tagline"), // Short description
  description: text("description"), // Full description
  sector: text("sector"), // Primary sector
  sectors: text("sectors").array(), // Multiple sectors
  level: text("level").notNull().default("L0"), // Current growth level L0-L8
  country: text("country"),
  city: text("city"),
  region: text("region"),
  website: text("website"),
  linkedinUrl: text("linkedin_url"),
  twitterHandle: text("twitter_handle"),
  logoUrl: text("logo_url"),
  foundedYear: integer("founded_year"),
  teamSize: text("team_size"), // "1-10", "11-50", "51-200", etc.
  fundingStage: text("funding_stage"), // "Pre-seed", "Seed", "Series A", etc.
  fundingAmount: text("funding_amount"), // Display string like "$5M"
  totalGleams: integer("total_gleams").default(0).notNull(),
  totalAlicorns: numeric("total_alicorns").default("0").notNull(),
  isVerified: boolean("is_verified").default(false).notNull(),
  isPublic: boolean("is_public").default(true).notNull(),
  isPremium: boolean("is_premium").default(false).notNull(),
  dataSource: text("data_source").default("user_generated").notNull(), // "user_generated", "crunchbase", "linkedin", etc.
  externalId: text("external_id"), // ID from external data source
  metadata: jsonb("metadata"), // Additional flexible data
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

export const insertEcosystemProfileSchema = createInsertSchema(ecosystemProfiles).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export type InsertEcosystemProfile = z.infer<typeof insertEcosystemProfileSchema>;
export type EcosystemProfile = typeof ecosystemProfiles.$inferSelect;

// Startup team members (for startups with founders/co-founders)
export const startupTeamMembers = pgTable("startup_team_members", {
  id: serial("id").primaryKey(),
  profileId: integer("profile_id").notNull().references(() => ecosystemProfiles.id, { onDelete: "cascade" }),
  name: text("name").notNull(),
  role: text("role").notNull(), // "Founder", "Co-Founder", "CEO", "CTO", etc.
  linkedinUrl: text("linkedin_url"),
  imageUrl: text("image_url"),
  bio: text("bio"),
  createdAt: timestamp("created_at").defaultNow().notNull(),
});

export const insertStartupTeamMemberSchema = createInsertSchema(startupTeamMembers).omit({
  id: true,
  createdAt: true,
});

export type InsertStartupTeamMember = z.infer<typeof insertStartupTeamMemberSchema>;
export type StartupTeamMember = typeof startupTeamMembers.$inferSelect;

// Ecosystem Activity Feed - platform activities
export const ecosystemActivity = pgTable("ecosystem_activity", {
  id: serial("id").primaryKey(),
  profileId: integer("profile_id").references(() => ecosystemProfiles.id, { onDelete: "cascade" }),
  sessionId: text("session_id"),
  activityType: text("activity_type").notNull(), // "joined", "level_up", "completed_assessment", "verified", etc.
  title: text("title").notNull(),
  description: text("description"),
  metadata: jsonb("metadata"), // Additional data like { oldLevel: "L1", newLevel: "L2" }
  isPublic: boolean("is_public").default(true).notNull(),
  createdAt: timestamp("created_at").defaultNow().notNull(),
});

export const insertEcosystemActivitySchema = createInsertSchema(ecosystemActivity).omit({
  id: true,
  createdAt: true,
});

export type InsertEcosystemActivity = z.infer<typeof insertEcosystemActivitySchema>;
export type EcosystemActivity = typeof ecosystemActivity.$inferSelect;

// Matching Preferences - what users are looking for
export const matchingPreferences = pgTable("matching_preferences", {
  id: serial("id").primaryKey(),
  profileId: integer("profile_id").notNull().references(() => ecosystemProfiles.id, { onDelete: "cascade" }),
  seekingEntityTypes: text("seeking_entity_types").array(), // Entity types they want to connect with
  seekingLevels: text("seeking_levels").array(), // Levels L0-L8
  seekingSectors: text("seeking_sectors").array(),
  seekingCountries: text("seeking_countries").array(),
  seekingFundingStages: text("seeking_funding_stages").array(),
  isActivelyMatching: boolean("is_actively_matching").default(true).notNull(),
  matchingNotes: text("matching_notes"), // Free-form text about what they're looking for
  createdAt: timestamp("created_at").defaultNow().notNull(),
  updatedAt: timestamp("updated_at").defaultNow().notNull(),
});

export const insertMatchingPreferencesSchema = createInsertSchema(matchingPreferences).omit({
  id: true,
  createdAt: true,
  updatedAt: true,
});

export type InsertMatchingPreferences = z.infer<typeof insertMatchingPreferencesSchema>;
export type MatchingPreferences = typeof matchingPreferences.$inferSelect;

// Match Connections - tracks matches between entities
export const matchConnections = pgTable("match_connections", {
  id: serial("id").primaryKey(),
  fromProfileId: integer("from_profile_id").notNull().references(() => ecosystemProfiles.id, { onDelete: "cascade" }),
  toProfileId: integer("to_profile_id").notNull().references(() => ecosystemProfiles.id, { onDelete: "cascade" }),
  status: text("status").notNull().default("pending"), // "pending", "accepted", "declined"
  matchScore: integer("match_score"), // Algorithm-calculated score 0-100
  matchReason: text("match_reason"), // Why they were matched
  initiatedBy: text("initiated_by"), // sessionId of who initiated
  createdAt: timestamp("created_at").defaultNow().notNull(),
  respondedAt: timestamp("responded_at"),
});

export const insertMatchConnectionSchema = createInsertSchema(matchConnections).omit({
  id: true,
  createdAt: true,
  respondedAt: true,
});

export type InsertMatchConnection = z.infer<typeof insertMatchConnectionSchema>;
export type MatchConnection = typeof matchConnections.$inferSelect;

// Cached News - stores fetched news articles
export const cachedNews = pgTable("cached_news", {
  id: serial("id").primaryKey(),
  title: text("title").notNull(),
  description: text("description"),
  url: text("url").notNull(),
  source: text("source"),
  imageUrl: text("image_url"),
  publishedAt: timestamp("published_at"),
  category: text("category"), // "startup", "funding", "tech", "ecosystem", etc.
  isActive: boolean("is_active").default(true).notNull(),
  fetchedAt: timestamp("fetched_at").defaultNow().notNull(),
});

export const insertCachedNewsSchema = createInsertSchema(cachedNews).omit({
  id: true,
  fetchedAt: true,
});

export type InsertCachedNews = z.infer<typeof insertCachedNewsSchema>;
export type CachedNews = typeof cachedNews.$inferSelect;
