const fs = require('fs');
const path = require('path');

const answerKeysPath = path.join(__dirname, '../client/src/lib/answerKeys.json');
const answerKeys = JSON.parse(fs.readFileSync(answerKeysPath, 'utf-8'));

const STAKEHOLDER = "Corporate Partner";

const allUpdates = {
  L0: {
    D1: {
      "1": "I have no awareness of problems; status quo dominates.",
      "2": "I have generic problems listed; they are vague.",
      "3": "I have identified clear business problems.",
      "4": "I have strategic problems with solutions ready.",
      "5": "I define transformative, sector-wide problems."
    },
    D2: {
      "1": "My problem definitions remain vague; unclear thinking.",
      "2": "I have superficial understanding; incomplete.",
      "3": "I have clear articulation with strategic insights.",
      "4": "I demonstrate deep understanding and systems thinking.",
      "5": "I articulate problems with transformative clarity."
    },
    D3: {
      "1": "I am closed; I prefer \"Not Invented Here\" (DIY).",
      "2": "I acknowledge external ideas but remain skeptical.",
      "3": "I am genuinely open; some barriers exist.",
      "4": "I actively seek external solutions; resources allocated.",
      "5": "I drive transformation through external solutions."
    },
    D4: {
      "1": "I have no engagement; I do not understand them.",
      "2": "I have token engagement; it is superficial.",
      "3": "I have real engagement; I learn from them.",
      "4": "I have deep engagement; it is mutually beneficial.",
      "5": "I am a legendary partner; founders seek me out."
    },
    D5: {
      "1": "I allocate no resources; token gestures only.",
      "2": "I allocate minimal resources; half-hearted.",
      "3": "I provide clear allocation with real commitment.",
      "4": "I provide significant resources structurally.",
      "5": "I allocate transformative resources; innovation-first."
    },
    D6: {
      "1": "I have generic insights based on public info.",
      "2": "I have some insights, but they are incomplete.",
      "3": "I share deep, proprietary industry insights.",
      "4": "I offer transformative insider perspectives.",
      "5": "I am an industry legend; I define the future."
    },
    D7: {
      "1": "I punish experiments; I am risk-averse.",
      "2": "I tolerate experiments but do not encourage them.",
      "3": "I celebrate learning and encourage trials.",
      "4": "I celebrate failure and structure learning.",
      "5": "I am legendary for enabling an experimentation culture."
    },
    D8: {
      "1": "I am paralyzed by bureaucracy; months to decide.",
      "2": "I am slow; weeks to make simple decisions.",
      "3": "I have a fast-track process for innovation.",
      "4": "I match startup velocity in decision-making.",
      "5": "I move lightning-fast; I remove all friction."
    },
    D9: {
      "1": "I punish risk-taking; strictly KPI-driven.",
      "2": "I offer no upside for innovation risk.",
      "3": "I have specific incentives for intrapreneurs.",
      "4": "I offer significant upside/equity-like rewards.",
      "5": "I have a founder-grade incentive structure."
    },
    EiR1: {
      "1": "I am doing this purely for PR/optics.",
      "2": "I am mostly focused on optics.",
      "3": "I have some substance behind the PR.",
      "4": "I am results-driven, not PR-driven.",
      "5": "I am purely impact-driven; substance first."
    },
    EiR2: {
      "1": "I reject anything we did not build.",
      "2": "I prefer internal builds strongly.",
      "3": "I am neutral; best solution wins.",
      "4": "I prefer buying/partnering for speed.",
      "5": "I am radically open ecosystem-wide."
    },
    EiR3: {
      "1": "I kill anything without quarterly ROI.",
      "2": "I struggle to fund long-term bets.",
      "3": "I protect innovation from quarterly cuts.",
      "4": "I have a ring-fenced long-term budget.",
      "5": "I have generational patience."
    },
    EiR4: {
      "1": "Failure here gets you fired.",
      "2": "Failure stalls your career.",
      "3": "Failure is accepted if learned from.",
      "4": "Failure is a badge of honor.",
      "5": "Innovation is the fast-track to C-suite."
    },
    EiR5: {
      "1": "Legal kills everything early.",
      "2": "Legal is a major bottleneck.",
      "3": "Legal is trained in startup norms.",
      "4": "Legal is a partner in speed.",
      "5": "Legal finds ways to say \"Yes.\""
    },
    EiR6: {
      "1": "Payment terms are Net-90+; kills startups.",
      "2": "Procurement is slow and rigid.",
      "3": "I have a simplified startup procurement path.",
      "4": "I pay startups instantly/Net-15.",
      "5": "Procurement is a competitive advantage."
    },
    EiR7: {
      "1": "I have no executive sponsor.",
      "2": "I have a weak/distracted sponsor.",
      "3": "I have a committed executive sponsor.",
      "4": "I have C-suite active participation.",
      "5": "The CEO is the chief innovation officer."
    },
    EiR8: {
      "1": "My best innovators are quitting.",
      "2": "I struggle to retain entrepreneurs.",
      "3": "I retain talent by giving them autonomy.",
      "4": "I attract entrepreneurial talent.",
      "5": "I am a magnet for the world's best builders."
    },
    EiR9: {
      "1": "Innovation is a side hobby.",
      "2": "Innovation is loosely coupled.",
      "3": "Innovation supports business units.",
      "4": "Innovation drives business unit strategy.",
      "5": "Innovation is the corporate strategy."
    }
  },
  L1: {
    D1: {
      "1": "I wait for startups to email me.",
      "2": "I look at general databases.",
      "3": "I scout based on specific problem theses.",
      "4": "I have a deep, active scouting network.",
      "5": "I find startups before they are known."
    },
    D2: {
      "1": "Business Units (BUs) ignore me.",
      "2": "BUs give vague problem statements.",
      "3": "BUs validate problems before I scout.",
      "4": "BUs actively partner in discovery.",
      "5": "BUs demand innovation; pull-based."
    },
    D3: {
      "1": "BUs refuse to pilot external tech.",
      "2": "BUs are hesitant to pilot.",
      "3": "BUs have budget set aside for pilots.",
      "4": "BUs compete to run pilots.",
      "5": "BUs run pilots at lightning speed."
    },
    D4: {
      "1": "I treat startups like big vendors.",
      "2": "I demand excessive documentation.",
      "3": "I respect startup resource constraints.",
      "4": "I minimize friction for startups.",
      "5": "I protect startups from my own complexity."
    },
    D5: {
      "1": "I am blind to disruption.",
      "2": "I see trends after they are mainstream.",
      "3": "I identify trends early.",
      "4": "I predict market shifts accurately.",
      "5": "I shape market trends."
    },
    D6: {
      "1": "NDAs take months.",
      "2": "NDAs take weeks.",
      "3": "NDAs take days.",
      "4": "Standardized, instant docs.",
      "5": "Zero-friction contracting."
    },
    D7: {
      "1": "I offer nothing but a logo.",
      "2": "I offer slow potential revenue.",
      "3": "I offer data, customers, and expertise.",
      "4": "I offer unfair market access.",
      "5": "I make startups market leaders."
    },
    D8: {
      "1": "I ghost startups.",
      "2": "I give vague rejections.",
      "3": "I give actionable feedback.",
      "4": "I connect them to other opportunities.",
      "5": "I am a trusted advisor even when saying no."
    },
    D9: {
      "1": "I have no plan for integration.",
      "2": "I figure it out later.",
      "3": "I have a hypothesis for scale.",
      "4": "I have a clear path to production.",
      "5": "I view every pilot as a potential M&A."
    },
    EiR1: {
      "1": "I buy tech looking for a problem.",
      "2": "I am excited by \"cool\" tech.",
      "3": "I start with the business problem.",
      "4": "I am rigorously problem-led.",
      "5": "I validate the problem before seeing tech."
    },
    EiR2: {
      "1": "Pilots define success; no scale plan.",
      "2": "Most pilots die after the POC.",
      "3": "I have a defined path out of pilot.",
      "4": "I have a high conversion to production.",
      "5": "I only pilot if I intend to scale."
    },
    EiR3: {
      "1": "BUs actively sabotage innovation.",
      "2": "BUs are indifferent.",
      "3": "BUs are cooperative.",
      "4": "BUs are champions.",
      "5": "BUs drive the innovation agenda."
    },
    EiR4: {
      "1": "Fear paralyzes all action.",
      "2": "Excessive compliance checks.",
      "3": "Balanced risk assessment.",
      "4": "Smart containment of risk.",
      "5": "We embrace reputational risk for growth."
    },
    EiR5: {
      "1": "I promise deals I cannot deliver.",
      "2": "I overhype internal interest.",
      "3": "I am transparent about decision power.",
      "4": "I only promise what I control.",
      "5": "My word is gold; startups trust me."
    },
    EiR6: {
      "1": "I give corporate advice to startups.",
      "2": "I give generic advice.",
      "3": "I give relevant industry context.",
      "4": "I provide game-changing mentorship.",
      "5": "I am a legendary mentor."
    },
    EiR7: {
      "1": "Data is locked; impossible to access.",
      "2": "Data access takes months.",
      "3": "Sandboxed data available.",
      "4": "API-first data access.",
      "5": "Seamless, secure data integration."
    },
    EiR8: {
      "1": "I meet just to \"learn\".",
      "2": "I ask for endless free consulting.",
      "3": "I respect their time/burn.",
      "4": "I add value in every meeting.",
      "5": "I accelerate their business regardless of deal."
    },
    EiR9: {
      "1": "I only see big-name startups.",
      "2": "I see what VCs send me.",
      "3": "I scout diverse/global sources.",
      "4": "I find hidden/stealth gems.",
      "5": "I have a proprietary deal flow engine."
    }
  },
  L2: {
    D1: {
      "1": "We build corporate projects, not startups.",
      "2": "We build slow, expensive MVPs.",
      "3": "We build lean, validated ventures.",
      "4": "We run a professional venture studio.",
      "5": "We build category-defining unicorns."
    },
    D2: {
      "1": "Legacy systems block everything.",
      "2": "We have some APIs, mostly legacy.",
      "3": "We have a modern integration layer.",
      "4": "We are cloud-native and API-first.",
      "5": "Our tech stack is a competitive moat."
    },
    D3: {
      "1": "No investment capability.",
      "2": "Ad hoc, strategic investments.",
      "3": "Dedicated CVC fund with thesis.",
      "4": "Top-tier CVC with financial returns.",
      "5": "Legendary CVC; smart money."
    },
    D4: {
      "1": "IP is locked forever.",
      "2": "Spin-outs are legally painful.",
      "3": "We have a process for spin-outs.",
      "4": "We actively spin out non-core IP.",
      "5": "We create ecosystems of spin-outs."
    },
    D5: {
      "1": "We treat them as vendors.",
      "2": "We try to collaborate; friction high.",
      "3": "We have co-creation frameworks.",
      "4": "We build joint IP effectively.",
      "5": "We seamlessly fuse teams."
    },
    D6: {
      "1": "Managed like IT projects (Waterfall).",
      "2": "Stage-gate with heavy docs.",
      "3": "VC-style metered funding.",
      "4": "Milestone-based growth boards.",
      "5": "Founder-friendly governance."
    },
    D7: {
      "1": "Staff cannot leave core roles.",
      "2": "HR makes transfers difficult.",
      "3": "Staff can second to ventures.",
      "4": "Seamless flow between core/venture.",
      "5": "Best talent fights to join ventures."
    },
    D8: {
      "1": "We hide our assets.",
      "2": "Hard to access assets.",
      "3": "We leverage brand/distribution.",
      "4": "We weaponize our scale for startups.",
      "5": "We provide unfair market dominance."
    },
    D9: {
      "1": "> 12 months.",
      "2": "6–12 months.",
      "3": "3–6 months.",
      "4": "1–3 months.",
      "5": "< 4 weeks."
    },
    EiR1: {
      "1": "We build everything; slow/expensive.",
      "2": "We try to build but lack skills.",
      "3": "We balance build versus buy.",
      "4": "We buy/partner for speed first.",
      "5": "We orchestrate; indifferent to source."
    },
    EiR2: {
      "1": "Legacy systems kill all new ideas.",
      "2": "Legacy makes integration painful.",
      "3": "We have workarounds for legacy.",
      "4": "Modern stack enables speed.",
      "5": "Tech stack is a competitive accelerator."
    },
    EiR3: {
      "1": "We have project managers, not builders.",
      "2": "We hire vendors to build everything.",
      "3": "We have a small internal dev team.",
      "4": "We have a high-density product team.",
      "5": "We attract ex-founders to build."
    },
    EiR4: {
      "1": "Committee approval needed for everything.",
      "2": "Governance is slow and risk-averse.",
      "3": "Governance is streamlined for pilot.",
      "4": "Governance facilitates speed.",
      "5": "Automated/Trust-based governance."
    },
    EiR5: {
      "1": "We demand 100% IP ownership always.",
      "2": "IP negotiations take months.",
      "3": "We have fair IP-sharing models.",
      "4": "We prioritize speed over IP control.",
      "5": "We create open IP ecosystems."
    },
    EiR6: {
      "1": "Projects always over budget/late.",
      "2": "Frequent overruns; scope creep.",
      "3": "Managed scope; predictable.",
      "4": "Lean builds; under budget.",
      "5": "Value-driven; ROI-focused spend."
    },
    EiR7: {
      "1": "We never kill projects; they drift.",
      "2": "Hard to kill once funded.",
      "3": "We have kill criteria.",
      "4": "We celebrate killing bad projects.",
      "5": "\"Fail fast\" is ingrained culture."
    },
    EiR8: {
      "1": "Sales blocks access to customers.",
      "2": "Access requires permission.",
      "3": "Builders have direct access.",
      "4": "Co-creation is the standard.",
      "5": "Customers are part of the team."
    },
    EiR9: {
      "1": "No MVPs allowed; brand risk.",
      "2": "MVPs must be perfect/polished.",
      "3": "MVPs allowed in sandboxes.",
      "4": "We launch unpolished betas often.",
      "5": "Brand is enhanced by experimentation."
    }
  },
  L3: {
    D1: {
      "1": "No path; pilot ends relationship.",
      "2": "Path is unclear/negotiated.",
      "3": "Standardized conversion path.",
      "4": "Automated triggers for scaling.",
      "5": "Guaranteed scale for winners."
    },
    D2: {
      "1": "We never expose to customers.",
      "2": "Limited exposure to friendlies.",
      "3": "Active sales channel integration.",
      "4": "Incentivized sales force sells it.",
      "5": "Seamless bundle with core product."
    },
    D3: {
      "1": "No marketing support.",
      "2": "We allow use of our logo.",
      "3": "Joint press releases.",
      "4": "Full co-marketing campaigns.",
      "5": "We make them a household name."
    },
    D4: {
      "1": "Procurement blocks scaling.",
      "2": "Volume discounts hard to negotiate.",
      "3": "Scaled master agreements.",
      "4": "Global framework agreements.",
      "5": "Vendor-of-choice status globally."
    },
    D5: {
      "1": "No metrics; anecdotal.",
      "2": "Vanity metrics (number of pilots).",
      "3": "ROI metrics (cost savings/revenue).",
      "4": "Strategic impact metrics.",
      "5": "Transformative P&L impact."
    },
    D6: {
      "1": "M&A is disconnected from innovation.",
      "2": "M&A is slow and destroys value.",
      "3": "M&A watches innovation pipeline.",
      "4": "Innovation feeds M&A targets.",
      "5": "Seamless acquire-and-integrate."
    },
    D7: {
      "1": "Local pilots only.",
      "2": "Hard to cross borders.",
      "3": "Regional rollout capability.",
      "4": "Global rollout infrastructure.",
      "5": "Instant global scale."
    },
    D8: {
      "1": "We take all the value.",
      "2": "We squeeze their margins.",
      "3": "Fair, standard terms.",
      "4": "Generous terms to fuel growth.",
      "5": "Ecosystem-first value sharing."
    },
    D9: {
      "1": "Employees resist new tech.",
      "2": "Forced adoption; low usage.",
      "3": "Champions drive adoption.",
      "4": "Widespread, enthusiastic use.",
      "5": "It becomes essential to daily work."
    },
    EiR1: {
      "1": "Pilots never convert; \"Innovation Theater\".",
      "2": "Conversion is rare and painful.",
      "3": "Defined path to conversion.",
      "4": "High conversion rate.",
      "5": "Pilots are essentially pre-contracts."
    },
    EiR2: {
      "1": "Procurement treats startups like IBM.",
      "2": "Process is too slow for startups.",
      "3": "Simplified path for <$100k.",
      "4": "Dedicated startup track.",
      "5": "Procurement is a growth enabler."
    },
    EiR3: {
      "1": "IT blocks integration post-pilot.",
      "2": "Integration takes 6–12 months.",
      "3": "Integration planned during pilot.",
      "4": "Plug-and-play integration.",
      "5": "Integration is seamless/API-first."
    },
    EiR4: {
      "1": "Sales ignores new/external tech.",
      "2": "Sales views it as a distraction.",
      "3": "Sales gets SPIFFs for innovation.",
      "4": "Quota relief/high comms for innovation.",
      "5": "Sales fights to sell the new stuff."
    },
    EiR5: {
      "1": "We measure PR/activity only.",
      "2": "We measure number of pilots.",
      "3": "We measure conversion rates.",
      "4": "We measure P&L impact.",
      "5": "We measure market transformation."
    },
    EiR6: {
      "1": "Compliance is a hard \"No\".",
      "2": "Compliance finds problems late.",
      "3": "Compliance involved early.",
      "4": "Compliance co-designs solutions.",
      "5": "Compliance creates competitive advantage."
    },
    EiR7: {
      "1": "Solution breaks at corporate scale.",
      "2": "Manual workarounds required.",
      "3": "Path to scale is tested.",
      "4": "Built for enterprise scale.",
      "5": "Infinitely scalable architecture."
    },
    EiR8: {
      "1": "We bankrupt startups with delays.",
      "2": "We stress partners financially.",
      "3": "We monitor partner health.",
      "4": "We invest/support partner cash flow.",
      "5": "We ensure partner profitability."
    },
    EiR9: {
      "1": "BUs actively block rollout.",
      "2": "BUs utilize \"pocket vetoes\".",
      "3": "BUs are neutral/cooperative.",
      "4": "BUs champion the rollout.",
      "5": "Innovation is demand-pull from BUs."
    }
  },
  L4: {
    D1: {
      "1": "Innovation is a cost center.",
      "2": "Innovation breaks even.",
      "3": "Innovation contributes to EBITDA.",
      "4": "Innovation drives material revenue.",
      "5": "Innovation saves/makes the company."
    },
    D2: {
      "1": "No ecosystem.",
      "2": "Vendor relationships only.",
      "3": "Partner ecosystem emerging.",
      "4": "Strong partner ecosystem.",
      "5": "Industry-defining ecosystem."
    },
    D3: {
      "1": "Innovation siloed.",
      "2": "Innovation tolerated.",
      "3": "Innovation embedded in BUs.",
      "4": "Innovation is a core capability.",
      "5": "Innovation is the culture."
    },
    D4: {
      "1": "No platform thinking.",
      "2": "Platforms are internal only.",
      "3": "Some platforms externalized.",
      "4": "Platform strategy in place.",
      "5": "Platform-first business model."
    },
    D5: {
      "1": "No international presence.",
      "2": "Domestic focus only.",
      "3": "Entering new markets.",
      "4": "Multi-regional presence.",
      "5": "Global innovation leader."
    },
    D6: {
      "1": "No M&A capability.",
      "2": "Opportunistic M&A.",
      "3": "Strategic M&A program.",
      "4": "M&A is growth engine.",
      "5": "M&A creates industry transformation."
    },
    D7: {
      "1": "No talent development.",
      "2": "Training is ad hoc.",
      "3": "Structured innovation training.",
      "4": "Innovation academy.",
      "5": "Talent magnet for innovators."
    },
    D8: {
      "1": "No repeat success.",
      "2": "One-hit wonder.",
      "3": "Emerging pattern of success.",
      "4": "Repeatable innovation engine.",
      "5": "Legendary innovation track record."
    },
    D9: {
      "1": "Core business declining.",
      "2": "Core stable; no growth.",
      "3": "Core and new balanced.",
      "4": "New business driving growth.",
      "5": "Next-gen business dominant."
    },
    EiR1: {
      "1": "Core business kills competitive innovation.",
      "2": "Core ignores innovation until it grows.",
      "3": "Core tolerates edge innovation.",
      "4": "Core partners with innovation.",
      "5": "Core cannibalizes itself to survive."
    },
    EiR2: {
      "1": "Innovation budget raided for core gaps.",
      "2": "Budget is unstable/cyclical.",
      "3": "Budget is protected.",
      "4": "Budget is ring-fenced and growing.",
      "5": "Innovation funds the core."
    },
    EiR3: {
      "1": "Best talent leaves to build startups.",
      "2": "High turnover in innovation teams.",
      "3": "We retain some key talent.",
      "4": "We attract top external talent.",
      "5": "We are the destination for founders."
    },
    EiR4: {
      "1": "Innovation buried in IT/Marketing.",
      "2": "Innovation is a siloed lab.",
      "3": "Innovation reports to C-suite.",
      "4": "Independent business units (IBUs).",
      "5": "Ambidextrous organization."
    },
    EiR5: {
      "1": "No upside; salary only.",
      "2": "Small bonuses; no equity.",
      "3": "Phantom equity/LTI offered.",
      "4": "Significant wealth creation opportunity.",
      "5": "Founder-like returns for success."
    },
    EiR6: {
      "1": "Corporate inertia halts growth.",
      "2": "Decision latency increases.",
      "3": "We fight to maintain speed.",
      "4": "Fast lanes for growth units.",
      "5": "We maintain Day 1 velocity."
    },
    EiR7: {
      "1": "Corporate brand hurts credibility.",
      "2": "Brand association is neutral.",
      "3": "Brand adds trust.",
      "4": "Brand is a massive accelerator.",
      "5": "Brand defines the category."
    },
    EiR8: {
      "1": "We integrate and kill culture immediately.",
      "2": "We lose the founders post-acquisition.",
      "3": "We keep them semi-independent.",
      "4": "We accelerate them post-acquisition.",
      "5": "We are the preferred acquirer."
    },
    EiR9: {
      "1": "Leadership focuses on quarterly earnings.",
      "2": "Innovation is a \"nice to have\".",
      "3": "Leadership reviews innovation often.",
      "4": "Leadership depends on innovation growth.",
      "5": "Innovation IS the leadership agenda."
    }
  },
  L5: {
    D1: {
      "1": "Innovation is cash drain.",
      "2": "Break-even innovation.",
      "3": "Profitable innovation.",
      "4": "High-margin innovation.",
      "5": "Innovation funds core R&D."
    },
    D2: {
      "1": "Inefficient innovation ops.",
      "2": "Some efficiency measures.",
      "3": "Lean innovation operations.",
      "4": "Optimized innovation stack.",
      "5": "World-class operational excellence."
    },
    D3: {
      "1": "No portfolio management.",
      "2": "Ad hoc portfolio.",
      "3": "Managed portfolio.",
      "4": "Optimized portfolio (Core/Adjacent/Transform).",
      "5": "Antifragile portfolio."
    },
    D4: {
      "1": "No synergy capture.",
      "2": "Accidental synergies.",
      "3": "Intentional synergy capture.",
      "4": "Synergy by design.",
      "5": "Synergy multiplier."
    },
    D5: {
      "1": "No integration efficiency.",
      "2": "Slow integrations.",
      "3": "Standard integration playbook.",
      "4": "Fast-track integrations.",
      "5": "Seamless \"bolt-on\" capability."
    },
    D6: {
      "1": "No shared services.",
      "2": "Duplicated efforts.",
      "3": "Some shared platforms.",
      "4": "Center of Excellence model.",
      "5": "Radical reuse and modularity."
    },
    D7: {
      "1": "No innovation accounting.",
      "2": "Traditional metrics only.",
      "3": "Innovation metrics emerging.",
      "4": "Comprehensive innovation accounting.",
      "5": "Real-time value tracking."
    },
    D8: {
      "1": "Bureaucratic processes.",
      "2": "Heavy governance.",
      "3": "Streamlined governance.",
      "4": "Automated governance.",
      "5": "Trust-based governance."
    },
    D9: {
      "1": "No asset leverage.",
      "2": "Underutilized assets.",
      "3": "Some asset leverage.",
      "4": "Strategic asset deployment.",
      "5": "Assets as competitive weapons."
    },
    EiR1: {
      "1": "We kill projects without immediate ROI.",
      "2": "We have unrealistic timelines.",
      "3": "We track leading indicators.",
      "4": "We have patient capital.",
      "5": "We value strategic option value."
    },
    EiR2: {
      "1": "Innovation is a black hole for cash.",
      "2": "High cost per success.",
      "3": "Benchmarked efficiency.",
      "4": "Capital-efficient engine.",
      "5": "Self-funding innovation machine."
    },
    EiR3: {
      "1": "All eggs in one basket.",
      "2": "Scattered, incoherent bets.",
      "3": "Balanced (Core/Adjacent/Transform).",
      "4": "Rigorous portfolio optimization.",
      "5": "Antifragile portfolio construction."
    },
    EiR4: {
      "1": "Innovation creates massive tech debt.",
      "2": "Integration costs eat margins.",
      "3": "Integration costs managed.",
      "4": "Efficient modular architecture.",
      "5": "Integration creates synergy value."
    },
    EiR5: {
      "1": "Data is siloed and inaccessible.",
      "2": "Manual data sharing.",
      "3": "Data lakes available.",
      "4": "API-first data sharing.",
      "5": "Data drives autonomous optimization."
    },
    EiR6: {
      "1": "Every BU builds the same tools.",
      "2": "Little sharing across BUs.",
      "3": "Center of Excellence exists.",
      "4": "Shared platforms/services.",
      "5": "Radical reuse and modularity."
    },
    EiR7: {
      "1": "We measure activity, not outcome.",
      "2": "We measure revenue only.",
      "3": "We measure profit and strategic value.",
      "4": "Comprehensive innovation accounting.",
      "5": "Automated value tracking."
    },
    EiR8: {
      "1": "Process is more important than product.",
      "2": "Heavy documentation required.",
      "3": "Lean processes implemented.",
      "4": "Automated governance.",
      "5": "Process empowers velocity."
    },
    EiR9: {
      "1": "We treat startups like strangers.",
      "2": "We offer cash only.",
      "3": "We offer some unfair advantages.",
      "4": "We leverage supply chain/brand fully.",
      "5": "We weaponize our scale for growth."
    }
  },
  L6: {
    D1: {
      "1": "No innovation leadership.",
      "2": "Manager-led innovation.",
      "3": "VP-led innovation.",
      "4": "C-suite innovation champion.",
      "5": "CEO is Chief Innovation Officer."
    },
    D2: {
      "1": "No innovation culture.",
      "2": "Pockets of innovation.",
      "3": "Innovation culture emerging.",
      "4": "Innovation culture strong.",
      "5": "Innovation is the culture."
    },
    D3: {
      "1": "No strategic innovation.",
      "2": "Reactive innovation.",
      "3": "Proactive innovation.",
      "4": "Strategic innovation.",
      "5": "Visionary innovation."
    },
    D4: {
      "1": "No ecosystem leadership.",
      "2": "Participant in ecosystems.",
      "3": "Active ecosystem member.",
      "4": "Ecosystem orchestrator.",
      "5": "Ecosystem architect."
    },
    D5: {
      "1": "No innovation org.",
      "2": "Siloed innovation team.",
      "3": "Integrated innovation function.",
      "4": "Networked innovation org.",
      "5": "Ambidextrous organization."
    },
    D6: {
      "1": "Fear-based culture.",
      "2": "Risk-averse culture.",
      "3": "Risk-tolerant culture.",
      "4": "Risk-embracing culture.",
      "5": "Fearless innovation culture."
    },
    D7: {
      "1": "No innovation talent.",
      "2": "Struggling to find talent.",
      "3": "Building innovation talent.",
      "4": "Strong innovation talent.",
      "5": "Magnet for world-class talent."
    },
    D8: {
      "1": "No innovation vision.",
      "2": "Vague innovation vision.",
      "3": "Clear innovation vision.",
      "4": "Compelling innovation vision.",
      "5": "Inspiring innovation vision."
    },
    D9: {
      "1": "No industry influence.",
      "2": "Follower.",
      "3": "Fast follower.",
      "4": "Industry leader.",
      "5": "Industry shaper."
    },
    EiR1: {
      "1": "No vision; confusion reigns.",
      "2": "Vision is a poster on the wall.",
      "3": "Vision is understood by leadership.",
      "4": "Vision guides all decisions.",
      "5": "Vision inspires the industry."
    },
    EiR2: {
      "1": "Failure is punished; people hide.",
      "2": "Risk-taking is discouraged.",
      "3": "\"Fail fast\" is spoken, not lived.",
      "4": "Psychological safety exists.",
      "5": "Fearlessness; audacity is rewarded."
    },
    EiR3: {
      "1": "Politics determine funding.",
      "2": "Success depends on who you know.",
      "3": "Meritocracy is attempted.",
      "4": "Ideas win, not titles.",
      "5": "Radical truth and transparency."
    },
    EiR4: {
      "1": "Employees resist all change.",
      "2": "\"Here we go again\" cynicism.",
      "3": "Change management is structured.",
      "4": "Organization is adaptable.",
      "5": "Organization craves change."
    },
    EiR5: {
      "1": "Silos war with each other.",
      "2": "Information hoarding.",
      "3": "Cross-functional collaboration.",
      "4": "Fluid, network-based teams.",
      "5": "Boundaryless organization."
    },
    EiR6: {
      "1": "Groupthink; no diversity.",
      "2": "Token diversity only.",
      "3": "Diverse teams in innovation.",
      "4": "Inclusive culture drives thought.",
      "5": "Cognitive diversity is a superpower."
    },
    EiR7: {
      "1": "Mistakes are buried.",
      "2": "Post-mortems are blame games.",
      "3": "We conduct blameless post-mortems.",
      "4": "Institutional learning loops.",
      "5": "We get smarter with every error."
    },
    EiR8: {
      "1": "Board demands quarterly focus.",
      "2": "Board allows small experiments.",
      "3": "Board supports strategy.",
      "4": "Board drives innovation agenda.",
      "5": "Board protects long-term bets."
    },
    EiR9: {
      "1": "\"We are a [legacy product] company.\"",
      "2": "\"We are trying to be digital.\"",
      "3": "\"We are a modern company.\"",
      "4": "\"We are innovators.\"",
      "5": "\"We invent the future.\""
    }
  },
  L7: {
    D1: {
      "1": "No industry impact.",
      "2": "Minor industry impact.",
      "3": "Noticeable industry impact.",
      "4": "Major industry impact.",
      "5": "Industry-defining impact."
    },
    D2: {
      "1": "No legacy.",
      "2": "Short-term legacy.",
      "3": "Medium-term legacy.",
      "4": "Long-term legacy.",
      "5": "Generational legacy."
    },
    D3: {
      "1": "No societal impact.",
      "2": "Minor societal impact.",
      "3": "Noticeable societal impact.",
      "4": "Major societal impact.",
      "5": "Transformative societal impact."
    },
    D4: {
      "1": "No inspiration.",
      "2": "Occasional inspiration.",
      "3": "Regular inspiration.",
      "4": "Consistent inspiration.",
      "5": "Legendary inspiration."
    },
    D5: {
      "1": "Local only.",
      "2": "Regional presence.",
      "3": "National presence.",
      "4": "International presence.",
      "5": "Global presence."
    },
    D6: {
      "1": "No disruption.",
      "2": "Incremental change.",
      "3": "Moderate disruption.",
      "4": "Major disruption.",
      "5": "Revolutionary disruption."
    },
    D7: {
      "1": "No industry leadership.",
      "2": "Emerging leader.",
      "3": "Established leader.",
      "4": "Dominant leader.",
      "5": "Legendary leader."
    },
    D8: {
      "1": "No ecosystem.",
      "2": "Basic ecosystem.",
      "3": "Growing ecosystem.",
      "4": "Strong ecosystem.",
      "5": "Legendary ecosystem."
    },
    D9: {
      "1": "No systemic change.",
      "2": "Minor systemic change.",
      "3": "Some systemic change.",
      "4": "Major systemic change.",
      "5": "Permanent systemic change."
    },
    EiR1: {
      "1": "Arrogance; \"we cannot fail.\"",
      "2": "Ignoring new competitors.",
      "3": "Vigilant but slow.",
      "4": "Paralyzed by success.",
      "5": "Only the paranoid survive."
    },
    EiR2: {
      "1": "We are being eaten alive.",
      "2": "We are losing market share.",
      "3": "We are defending share.",
      "4": "We are the disruptor.",
      "5": "We disrupt ourselves continuously."
    },
    EiR3: {
      "1": "We are a bully; partners leave.",
      "2": "Transactional relationships.",
      "3": "Good partner ecosystem.",
      "4": "Partners grow with us.",
      "5": "We nourish the entire ecosystem."
    },
    EiR4: {
      "1": "We fight regulators.",
      "2": "We are constantly fined.",
      "3": "We comply.",
      "4": "We help shape policy.",
      "5": "We set the global standard."
    },
    EiR5: {
      "1": "Talent leaves for younger firms.",
      "2": "We are seen as a \"retirement home.\"",
      "3": "We retain stable performers.",
      "4": "We are a talent academy.",
      "5": "We have the highest talent density."
    },
    EiR6: {
      "1": "Profit over everything; scandals.",
      "2": "Ethical grey areas ignored.",
      "3": "Ethical codes followed.",
      "4": "Ethics by design.",
      "5": "Moral leadership in industry."
    },
    EiR7: {
      "1": "Bureaucracy strangling value.",
      "2": "Too complex to move.",
      "3": "Complexity managed.",
      "4": "Radical simplification.",
      "5": "Complexity abstracted away."
    },
    EiR8: {
      "1": "Focus is only on stock price.",
      "2": "Purpose is marketing fluff.",
      "3": "Purpose is stated.",
      "4": "Purpose guides strategy.",
      "5": "Purpose is the heartbeat."
    },
    EiR9: {
      "1": "A cautionary tale of decline.",
      "2": "A faded giant.",
      "3": "A stable institution.",
      "4": "An industry shaper.",
      "5": "A blueprint for the future."
    }
  },
  L8: {
    D1: {
      "1": "Extractive model.",
      "2": "Compliance-only.",
      "3": "Responsible practices.",
      "4": "Stewardship mindset.",
      "5": "Regenerative model."
    },
    D2: {
      "1": "Shareholder primacy only.",
      "2": "Some stakeholder consideration.",
      "3": "Balanced stakeholder value.",
      "4": "Stakeholder-first.",
      "5": "Ecosystem value creation."
    },
    D3: {
      "1": "Extractive sustainability.",
      "2": "Compliance sustainability.",
      "3": "Proactive sustainability.",
      "4": "Sustainable by design.",
      "5": "Regenerative and net-positive."
    },
    D4: {
      "1": "No social impact.",
      "2": "CSR/Charity.",
      "3": "Strategic social investment.",
      "4": "Integrated social impact.",
      "5": "Transformative social change."
    },
    D5: {
      "1": "Opaque governance.",
      "2": "Compliance governance.",
      "3": "Transparent governance.",
      "4": "Best-practice governance.",
      "5": "Stakeholder governance."
    },
    D6: {
      "1": "Ethics as liability.",
      "2": "Ethics as compliance.",
      "3": "Ethics as values.",
      "4": "Ethics by design.",
      "5": "Moral leadership."
    },
    D7: {
      "1": "Community extraction.",
      "2": "Transactional community.",
      "3": "Community partnership.",
      "4": "Community co-creation.",
      "5": "Community empowerment."
    },
    D8: {
      "1": "Quarterly thinking.",
      "2": "Annual thinking.",
      "3": "5–10 year thinking.",
      "4": "Multi-decade thinking.",
      "5": "100-year stewardship."
    },
    D9: {
      "1": "Values as marketing.",
      "2": "Values stated.",
      "3": "Values practiced.",
      "4": "Values enforced.",
      "5": "Values are the strategy."
    },
    EiR1: {
      "1": "Pure PR; zero substance.",
      "2": "Compliance minimums only.",
      "3": "Authentic initiatives.",
      "4": "Sustainability is core strategy.",
      "5": "We are net-positive/regenerative."
    },
    EiR2: {
      "1": "We exploit cheap labor/resources.",
      "2": "We ignore wealth gaps.",
      "3": "We pay fair wages.",
      "4": "We create broad wealth.",
      "5": "We systematically reduce inequality."
    },
    EiR3: {
      "1": "Public enemy; distrusted.",
      "2": "Skepticism is high.",
      "3": "Generally trusted.",
      "4": "Highly trusted brand.",
      "5": "A pillar of societal trust."
    },
    EiR4: {
      "1": "Trapped in quarterly cycle.",
      "2": "1–3 year horizon max.",
      "3": "5–10 year planning.",
      "4": "Multi-decade vision.",
      "5": "100-year stewardship mindset."
    },
    EiR5: {
      "1": "We dump costs on society.",
      "2": "We minimize liability.",
      "3": "We mitigate harm.",
      "4": "We internalize externalities.",
      "5": "We solve societal problems."
    },
    EiR6: {
      "1": "Dictatorial/Opaque.",
      "2": "Defensive governance.",
      "3": "Transparent governance.",
      "4": "Stakeholder governance.",
      "5": "Radical accountability."
    },
    EiR7: {
      "1": "We are obsolete/harmful.",
      "2": "We are a utility.",
      "3": "We are useful.",
      "4": "We are important.",
      "5": "We are essential to human progress."
    },
    EiR8: {
      "1": "We extract from communities.",
      "2": "We give charity (transactional).",
      "3": "We are a good citizen.",
      "4": "We co-create with community.",
      "5": "We empower communities to thrive."
    },
    EiR9: {
      "1": "No succession; chaos ahead.",
      "2": "Unclear succession.",
      "3": "Planned succession.",
      "4": "Strong institutional memory.",
      "5": "The mission transcends the leaders."
    }
  }
};

let updateCount = 0;
let missingCount = 0;

for (const [levelId, dimensions] of Object.entries(allUpdates)) {
  for (const [dimId, grades] of Object.entries(dimensions)) {
    if (!answerKeys[levelId]) {
      console.log(`Missing level: ${levelId}`);
      missingCount++;
      continue;
    }
    if (!answerKeys[levelId].questions) {
      console.log(`Missing questions in level: ${levelId}`);
      missingCount++;
      continue;
    }
    if (!answerKeys[levelId].questions[dimId]) {
      console.log(`Missing dimension: ${levelId} ${dimId}`);
      missingCount++;
      continue;
    }
    if (!answerKeys[levelId].questions[dimId].stakeholders) {
      answerKeys[levelId].questions[dimId].stakeholders = {};
    }
    if (!answerKeys[levelId].questions[dimId].stakeholders[STAKEHOLDER]) {
      answerKeys[levelId].questions[dimId].stakeholders[STAKEHOLDER] = {
        question: "",
        grades: {}
      };
    }
    
    const stakeholderData = answerKeys[levelId].questions[dimId].stakeholders[STAKEHOLDER];
    if (!stakeholderData.grades) {
      stakeholderData.grades = {};
    }
    
    for (const [grade, answer] of Object.entries(grades)) {
      stakeholderData.grades[grade] = answer;
    }
    updateCount++;
    
    console.log(`Updated ${levelId} ${dimId} (5 grades)`);
  }
}

fs.writeFileSync(answerKeysPath, JSON.stringify(answerKeys, null, 2), 'utf-8');

console.log(`\n✅ Updated ${updateCount} Corporate Partner entries with all 5 grades.`);
console.log(`⚠️ Missing entries: ${missingCount}`);
console.log(`Total individual grade updates: ${updateCount * 5}`);
console.log(`File saved to: ${answerKeysPath}`);
